//
//  AHKPhone_Address.m
//  AH-K3001V Utilities
//
//  Created by FUJIDANA on 06/04/23.
//  Copyright 2006 FUJIDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "AHKPhone_Address.h"
#import "AHKProgressing.h"
#import "AHKPerson.h"

@implementation AHKPhone (Address)

// --- read address cards from cellphone

- (BOOL)readPeople:(NSArray **)people password:(NSString *)password progress:(id <AHKProgressing>)progress error:(NSError **)errorPtr
{
	UInt8 buffer[kReadBufferSize];
	
	// begin progress
	[progress beginProgressWithMessage:NSLocalizedStringFromTable(@"Sending Password...", @"AHKABLocalizable", @"phone.message.password")];
	
	// log in.
	if ([self openError:errorPtr] == NO) goto ERROR;
	if ([self loginWithPassword:password error:errorPtr] == NO) goto ERROR;
	
	// count number of address cards
	if ([self sendCommand:2 subCommand:0x0301 parameter:"" error:errorPtr] == NO) goto ERROR;
	if ([self receiveData:buffer maxLength:sizeof(buffer) additionally:YES error:errorPtr] == -1) goto ERROR;
	const int count = atoi((char *)buffer + 12);
	
	// prepare array
	NSMutableArray *array = [NSMutableArray arrayWithCapacity:count];
	[progress startDeterminateProgressWithMaxValue:count
										   message:NSLocalizedStringFromTable(@"Receiving Address Book...", @"AHKABLocalizable", @"phone.message.readPeople")];
	if(count > 0) {
		// begin to read address cards.
		if ([self sendCommand:3 subCommand:0x0302 parameter:"" error:errorPtr] == NO) goto ERROR;
		
		int n;
		for (n = 0; n < count; n++) {
			// Now safe to abort communication, so check whether user requested to abort progress
			if ([progress isRequestedToAbort]) {
				if (errorPtr) {
					NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedStringFromTable(@"Reading address cards was interrupted by user. The database of your computer is not updated.", @"AHKABLocalizable", @"error.userCancelledWhileReadingAddresses.description")
																		 forKey:NSLocalizedDescriptionKey];
					*errorPtr = [NSError errorWithDomain:AHKErrorDomain
													code:AHKPhoneUserCancelledError
												userInfo:userInfo];
				}
				goto ERROR;
			}
			
			// Read an address card.
			SInt32 restBytes = [self receiveData:buffer maxLength:sizeof(buffer) additionally:YES error:errorPtr];
			if (restBytes == -1) goto ERROR;
			if (buffer[3] != 0x03 || buffer[9] != 0x03 || buffer[11] != 0x02) {
				if (errorPtr) {
					NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedStringFromTable(@"Illegal format data were found in receiving an address card.", @"AHKABLocalizable", @"error.InapplicableAddressDataReceived.description")
																		 forKey:NSLocalizedDescriptionKey];
					*errorPtr = [NSError errorWithDomain:AHKErrorDomain
													code:AHKPhoneInapplicableDataReceivedError
												userInfo:userInfo];
				}
				goto ERROR;
				// [NSException raise:kAirHPhoneException format:@"Illegal format data were found in receiving addresses."];
			}
			if ([self sendCommand:0x10 subCommand:0x0910 parameter:"0" error:errorPtr] == NO) goto ERROR;
			
			// Parse adress card data obtained above (regarding the data as VCARD format)
			NSData *rawData = [NSData dataWithBytes:buffer+12 length:restBytes-12];
			
			AHKPerson *person = [[AHKPerson alloc] initWithVCardRepresentation:rawData format:AHKKyoponVCardFormat];
			if (person) {
				[array addObject:person];
				[person release];
			} else {
				// #warning illegal error received
			}
			
			// refresh progress window.
			[progress incrementProgressBarBy:1];
		}
	}
	[progress endProgress];
	if ([self closeError:errorPtr] == NO) return NO;
	*people = array;
	return YES;
	
ERROR:
	[progress endProgress];
	[self closeError:NULL];
	return NO;
}

// --- AIR-EDGE PHONEへアドレス帳を書き込む。

- (BOOL)writePeople:(NSArray *)array withPassword:(NSString *)password progress:(id <AHKProgressing>)progress error:(NSError **)errorPtr
{
	// アドレスカードの最大個数をチェックする。
	const SInt32 count = [array count];
	if(count > kAHKPersonMaximumNumber) {
		if (errorPtr) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedStringFromTable(@"Too many address cards exist. Up to 500 address cards can be registered.", @"AHKABLocalizable", @"error.tooManyCardsSending.description")
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:AHKErrorDomain
											code:AHKPhoneTooManyItemsSendingError
										userInfo:userInfo];
		}
		return NO;
//		[NSException raise:kAirHPhoneException format:@"Too many address cards exist. Up to 500 address cards can be registered."];
	}
	
	BOOL  startAppendMode = NO;
	
//	@try {
	UInt8  readBuffer[kReadBufferSize];
	UInt8  writeBuffer[kWriteBufferSize];
	SInt32 n;
	
	[progress beginProgressWithMessage:NSLocalizedStringFromTable(@"Sending Password...", @"AHKABLocalizable", @"phone.message.password")];
	
	// log in
	if ([self openError:errorPtr] == NO) goto ERROR;
	
	// send password
	if([self loginWithPassword:password error:errorPtr] == NO) goto ERROR;
	
	// remove old address cards in cellphone
	if ([self sendCommand:3 subCommand:0x0304 parameter:"10" error:errorPtr] == NO) goto ERROR;
	if ([self receiveData:readBuffer maxLength:sizeof(readBuffer) additionally:YES error:errorPtr] == -1) goto ERROR;
	
	// begin to send address cards
	if ([self sendCommand:3 subCommand:0x0305 parameter:"" error:errorPtr] == NO) goto ERROR;
	if ([self receiveData:readBuffer maxLength:sizeof(readBuffer) additionally:YES error:errorPtr] == -1) goto ERROR;
	startAppendMode = YES;
	
	[progress startDeterminateProgressWithMaxValue:count message:NSLocalizedStringFromTable(@"Sending Address Book...", @"AHKABLocalizable", @"phone.message.writePeople")];
	for (n = 0; n < count; n++) {
		if ([progress isRequestedToAbort]) {
			if (errorPtr) {
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedStringFromTable(@"Writing is interrupted by user. The database of your cellular phone is not incomplete. Remind that if you read data from your cellular phone the database in your computer might be lost.", @"AHKABLocalizable", @"error.userCancelledWhileWriting.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:AHKErrorDomain
												code:AHKPhoneUserCancelledError
											userInfo:userInfo];
			}
			goto ERROR;
		}
		AHKPerson	*person	= [array objectAtIndex:n];
		NSData		*data	= [person vCardRepresentation];
		
		[data getBytes:writeBuffer];
		
		// send a address card
		if ([self sendCommand:3 subCommand:0x0303 parameter:writeBuffer length:[data length] error:errorPtr] == NO) goto ERROR;
		if ([self receiveData:readBuffer maxLength:sizeof(readBuffer) additionally:YES error:errorPtr] == -1) goto ERROR;
		if(strncmp((char *)(readBuffer + 12), "0", 1)) {
			if (errorPtr) {
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedStringFromTable(@"The AIR-EDGE PHONE didn't return the success code in sending address card.", @"AHKABLocalizable", @"error.failToSendCard.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:AHKErrorDomain
												code:AHKPhoneMalformedItemSentError
											userInfo:userInfo];
			}
			goto ERROR;
//			[NSException raise:kAirHPhoneException format:@"The AIR-EDGE PHONE didn't return the success code."];
		}
		// refresh progress status
		[progress incrementProgressBarBy:1];
	}
	
	if(startAppendMode) {
		// finish sending address cards
		UInt8 readBuffer[kReadBufferSize];
		if ([self sendCommand:3 subCommand:0x0306 parameter:"" error:errorPtr] == NO) goto ERROR;
		if ([self receiveData:readBuffer maxLength:sizeof(readBuffer) additionally:YES error:errorPtr] == -1) goto ERROR;
	}
	[progress endProgress];
	[self closeError:errorPtr];
	return YES;
	
//	} @catch(NSException *exception) {
//		[progress notifyException:exception];
//		@throw;
//	} @finally {
//		if(startAppendMode) {
//			// アドレスカードの追加を終了する。
//			UInt8 readBuffer[kReadBufferSize];
//			[self sendCommand:3 subCommand:0x0306 parameter:""];
//			[self receiveData:readBuffer maxLength:sizeof(readBuffer) additionally:YES];
//		}
//		//[progress closeProgress];
//		[progress endProgress];
//		[self close];
//	}
	return YES;
ERROR:
	if(startAppendMode) {
		// アドレスカードの追加を終了する。
		UInt8 readBuffer[kReadBufferSize];
		[self sendCommand:3 subCommand:0x0306 parameter:"" error:NULL];
		[self receiveData:readBuffer maxLength:sizeof(readBuffer) additionally:YES error:NULL];
	}
	[progress endProgress];
	[self closeError:NULL];
	return NO;
}

@end
